<?php
/*
Plugin Name: Hitokoto Manager & API (Advanced)
Description: 自建一言 API + taxonomy + 导入/导出 + 短缓存
Version: 0.2.0
Author: gomkiri
*/
if (!defined('ABSPATH')) exit;

/** 可调缓存 TTL（秒） */
define('HITOKOTO_RANDOM_TTL', 30);   // 单条随机结果短缓存
define('HITOKOTO_POOL_TTL',   300);  // ID池缓存

function hitokoto_random_ttl() { return apply_filters('hitokoto_random_ttl', HITOKOTO_RANDOM_TTL); }
function hitokoto_pool_ttl()   { return apply_filters('hitokoto_pool_ttl',   HITOKOTO_POOL_TTL); }

function hitokoto_strlen($str) { return function_exists('mb_strlen') ? mb_strlen($str) : strlen($str); }

/** 注册 CPT 和 taxonomy */
add_action('init', function() {
  // CPT: 一言
  register_post_type('hitokoto', [
    'labels' => [
      'name' => '一言',
      'singular_name' => '一言',
      'add_new' => '新增一言',
      'add_new_item' => '新增一言',
      'edit_item' => '编辑一言',
      'new_item' => '新一言',
      'view_item' => '查看一言',
      'search_items' => '搜索一言',
      'not_found' => '未找到一言',
      'menu_name' => '一言管理',
    ],
    'public' => false,
    'show_ui' => true,
    'show_in_menu' => true,
    'menu_icon' => 'dashicons-format-quote',
    'supports' => ['title', 'editor'],
    'capability_type' => 'post',
  ]);

  // taxonomy: 类型（非层级）
  register_taxonomy('hitokoto_type', ['hitokoto'], [
    'labels' => [
      'name' => '类型',
      'singular_name' => '类型',
      'search_items' => '搜索类型',
      'all_items' => '所有类型',
      'edit_item' => '编辑类型',
      'update_item' => '更新类型',
      'add_new_item' => '新增类型',
      'new_item_name' => '新类型',
      'menu_name' => '类型',
    ],
    'public' => false,
    'show_ui' => true,
    'show_admin_column' => true,
    'hierarchical' => false,
    'rewrite' => false,
  ]);
});

/** 元字段：来源/作者/语言 */
add_action('add_meta_boxes', function() {
  add_meta_box('hitokoto_meta', '一言信息', function($post) {
    wp_nonce_field('hitokoto_meta', 'hitokoto_meta_nonce');
    $from = get_post_meta($post->ID, 'hitokoto_from', true);
    $from_who = get_post_meta($post->ID, 'hitokoto_from_who', true);
    $lang = get_post_meta($post->ID, 'hitokoto_lang', true);
    echo '<p><label>来源（from）</label><br><input type="text" name="hitokoto_from" value="' . esc_attr($from) . '" style="width:100%"></p>';
    echo '<p><label>作者（from_who）</label><br><input type="text" name="hitokoto_from_who" value="' . esc_attr($from_who) . '" style="width:100%"></p>';
    echo '<p><label>语言（lang）</label><br><input type="text" name="hitokoto_lang" value="' . esc_attr($lang) . '" style="width:100%"></p>';
    echo '<p>提示：正文即为一言内容；标题可用于内部检索或留空。类型请在右侧 taxonomy 设置。</p>';
  }, 'hitokoto', 'side', 'default');
});

add_action('save_post_hitokoto', function($post_id) {
  if (!isset($_POST['hitokoto_meta_nonce']) || !wp_verify_nonce($_POST['hitokoto_meta_nonce'], 'hitokoto_meta')) return;
  if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
  if (!current_user_can('edit_post', $post_id)) return;

  foreach (['hitokoto_from', 'hitokoto_from_who', 'hitokoto_lang'] as $f) {
    $v = isset($_POST[$f]) ? sanitize_text_field($_POST[$f]) : '';
    update_post_meta($post_id, $f, $v);
  }
  // 保存内容哈希用于去重
  $p = get_post($post_id);
  if ($p) {
    $hash = md5(trim(wp_strip_all_tags($p->post_content)));
    update_post_meta($post_id, 'hitokoto_hash', $hash);
  }
});

/** 列表页：来源/作者/语言列 */
add_filter('manage_hitokoto_posts_columns', function($cols) {
  $cols['from'] = '来源';
  $cols['from_who'] = '作者';
  $cols['lang'] = '语言';
  return $cols;
});
add_action('manage_hitokoto_posts_custom_column', function($col, $post_id) {
  if ($col === 'from') echo esc_html(get_post_meta($post_id, 'hitokoto_from', true));
  if ($col === 'from_who') echo esc_html(get_post_meta($post_id, 'hitokoto_from_who', true));
  if ($col === 'lang') echo esc_html(get_post_meta($post_id, 'hitokoto_lang', true));
}, 10, 2);

/** 列表页：按类型筛选 */
add_action('restrict_manage_posts', function() {
  $screen = get_current_screen();
  if (!$screen || $screen->post_type !== 'hitokoto') return;
  wp_dropdown_categories([
    'taxonomy' => 'hitokoto_type',
    'show_option_all' => '全部类型',
    'name' => 'hitokoto_type',
    'orderby' => 'name',
    'selected' => isset($_GET['hitokoto_type']) ? $_GET['hitokoto_type'] : '',
    'hide_empty' => false,
  ]);
});
add_filter('parse_query', function($query) {
  global $pagenow;
  if ($pagenow === 'edit.php' && isset($_GET['post_type']) && $_GET['post_type'] === 'hitokoto' && !empty($_GET['hitokoto_type'])) {
    $term = get_term_by('id', (int)$_GET['hitokoto_type'], 'hitokoto_type');
    if ($term) {
      $query->query_vars['tax_query'] = [[
        'taxonomy' => 'hitokoto_type',
        'field' => 'slug',
        'terms' => $term->slug
      ]];
    }
  }
});

/** 缓存键 */
function hitokoto_build_pool_key($type_slug, $lang)  { return 'hitokoto_pool_'  . md5($type_slug . '|' . $lang); }
function hitokoto_build_random_key($type_slug, $lang){ return 'hitokoto_rand_'  . md5($type_slug . '|' . $lang); }

/** 获取 ID 池（带缓存） */
function hitokoto_get_id_pool($type_slug, $lang) {
  $key = hitokoto_build_pool_key($type_slug, $lang);
  $ids = get_transient($key);
  if ($ids === false) {
    $tax_query = [];
    if (!empty($type_slug)) {
      $tax_query[] = [
        'taxonomy' => 'hitokoto_type',
        'field' => 'slug',
        'terms' => $type_slug
      ];
    }
    $meta_query = [];
    if (!empty($lang)) {
      $meta_query[] = [
        'key' => 'hitokoto_lang',
        'value' => $lang,
        'compare' => '='
      ];
    }
    $q = new WP_Query([
      'post_type' => 'hitokoto',
      'post_status' => 'publish',
      'fields' => 'ids',
      'posts_per_page' => -1,
      'no_found_rows' => true,
      'tax_query' => $tax_query,
      'meta_query' => $meta_query
    ]);
    $ids = $q->posts;
    set_transient($key, $ids, hitokoto_pool_ttl());
  }
  return is_array($ids) ? $ids : [];
}

/** 挑选随机条目（带短缓存） */
function hitokoto_pick_random($type_slug, $lang, $disable_cache = false) {
  $cache_key = hitokoto_build_random_key($type_slug, $lang);
  if (!$disable_cache) {
    $cached = get_transient($cache_key);
    if ($cached !== false) return $cached;
  }
  $ids = hitokoto_get_id_pool($type_slug, $lang);
  if (empty($ids)) return new WP_Error('not_found', '没有匹配的一言', ['status' => 404]);

  $post_id = $ids[array_rand($ids)];
  $p = get_post($post_id);
  $content = trim(wp_strip_all_tags($p->post_content));
  $terms = wp_get_post_terms($post_id, 'hitokoto_type');
  $type_slug_actual = isset($terms[0]) ? $terms[0]->slug : '';
  $from     = get_post_meta($post_id, 'hitokoto_from', true);
  $from_who = get_post_meta($post_id, 'hitokoto_from_who', true);
  $lang_val = get_post_meta($post_id, 'hitokoto_lang', true);

  $resp = [
    'id'       => (int)$post_id,
    'hitokoto' => $content,
    'from'     => $from ?: '',
    'from_who' => $from_who ?: '',
    'type'     => $type_slug_actual,
    'lang'     => $lang_val ?: ($lang ?: ''),
    'length'   => hitokoto_strlen($content),
  ];
  set_transient($cache_key, $resp, hitokoto_random_ttl());
  return $resp;
}

/** REST 路由 */
add_action('rest_api_init', function() {
  register_rest_route('hitokoto/v1', '/random', [
    'methods' => 'GET',
    'callback' => 'hitokoto_api_random',
    'permission_callback' => '__return_true',
  ]);
  register_rest_route('hitokoto/v1', '/text', [
    'methods' => 'GET',
    'callback' => 'hitokoto_api_text',
    'permission_callback' => '__return_true',
  ]);
  register_rest_route('hitokoto/v1', '/export', [
    'methods' => 'GET',
    'callback' => 'hitokoto_api_export',
    'permission_callback' => function() {
      return is_user_logged_in() && current_user_can('manage_options');
    },
  ]);
});

/** 随机 JSON */
function hitokoto_api_random(WP_REST_Request $request) {
  $type = sanitize_text_field($request->get_param('type')); // taxonomy slug
  $lang = sanitize_text_field($request->get_param('lang'));
  $disable_cache = (bool)$request->get_param('nocache');
  $res = hitokoto_pick_random($type, $lang, $disable_cache);
  if (is_wp_error($res)) return $res;
  return rest_ensure_response($res);
}

/** 纯文本端点 */
function hitokoto_api_text(WP_REST_Request $request) {
  $type = sanitize_text_field($request->get_param('type'));
  $lang = sanitize_text_field($request->get_param('lang'));
  $disable_cache = (bool)$request->get_param('nocache');
  $res = hitokoto_pick_random($type, $lang, $disable_cache);
  if (is_wp_error($res)) {
    $response = new WP_REST_Response('暂无一言', 404);
    $response->set_headers(['Content-Type' => 'text/plain; charset=UTF-8']);
    return $response;
  }
  $response = new WP_REST_Response($res['hitokoto'], 200);
  $response->set_headers(['Content-Type' => 'text/plain; charset=UTF-8']);
  return $response;
}

/** 导出（REST） */
function hitokoto_api_export(WP_REST_Request $req) {
  $format = $req->get_param('format') ?: 'json';
  $type   = sanitize_text_field($req->get_param('type'));
  $lang   = sanitize_text_field($req->get_param('lang'));
  $ids = hitokoto_get_id_pool($type, $lang);

  if ($format === 'csv') {
    $lines = ['hitokoto,from,from_who,type,lang'];
    foreach ($ids as $id) {
      $p = get_post($id);
      $content = str_replace(["\r","\n"], ' ', trim(wp_strip_all_tags($p->post_content)));
      $from     = get_post_meta($id, 'hitokoto_from', true);
      $from_who = get_post_meta($id, 'hitokoto_from_who', true);
      $lang_val = get_post_meta($id, 'hitokoto_lang', true);
      $terms = wp_get_post_terms($id, 'hitokoto_type');
      $type_slug_actual = isset($terms[0]) ? $terms[0]->slug : '';
      $row = [
        $content,
        $from ?: '',
        $from_who ?: '',
        $type_slug_actual,
        $lang_val ?: ''
      ];
      $row = array_map(function($v){ return '"' . str_replace('"','""',$v) . '"'; }, $row);
      $lines[] = implode(',', $row);
    }
    $csv = implode("\r\n", $lines);
    $response = new WP_REST_Response($csv, 200);
    $response->set_headers(['Content-Type'=>'text/csv; charset=UTF-8']);
    return $response;
  } else {
    $data = [];
    foreach ($ids as $id) {
      $p = get_post($id);
      $content = trim(wp_strip_all_tags($p->post_content));
      $from     = get_post_meta($id, 'hitokoto_from', true);
      $from_who = get_post_meta($id, 'hitokoto_from_who', true);
      $lang_val = get_post_meta($id, 'hitokoto_lang', true);
      $terms = wp_get_post_terms($id, 'hitokoto_type');
      $type_slug_actual = isset($terms[0]) ? $terms[0]->slug : '';
      $data[] = [
        'id'       => (int)$id,
        'hitokoto' => $content,
        'from'     => $from ?: '',
        'from_who' => $from_who ?: '',
        'type'     => $type_slug_actual,
        'lang'     => $lang_val ?: '',
        'length'   => hitokoto_strlen($content),
      ];
    }
    return rest_ensure_response($data);
  }
}

/** 管理页：导入/导出 */
add_action('admin_menu', function() {
  add_submenu_page(
    'edit.php?post_type=hitokoto',
    '导入/导出',
    '导入/导出',
    'manage_options',
    'hitokoto-import-export',
    'hitokoto_render_import_export'
  );
});
function hitokoto_render_import_export() {
  if (!current_user_can('manage_options')) return;
  ?>
  <div class="wrap">
    <h1>一言 导入/导出</h1>
    <?php if (isset($_GET['import_result'])): ?>
      <div class="notice notice-success"><p><?php echo esc_html($_GET['import_result']); ?></p></div>
    <?php endif; ?>
    <h2>导入</h2>
    <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data">
      <?php wp_nonce_field('hitokoto_import','hitokoto_import_nonce'); ?>
      <input type="hidden" name="action" value="hitokoto_import">
      <p><input type="file" name="hitokoto_file" accept=".json,.csv,.txt" required></p>
      <p>支持：JSON（数组或逐行 JSON）或 CSV（UTF-8，首行：hitokoto,from,from_who,type,lang）。</p>
      <p><label><input type="checkbox" name="replace" value="1"> 相同内容（按哈希）则更新原条目</label></p>
      <p><button class="button button-primary" type="submit">开始导入</button></p>
    </form>
    <h2>导出</h2>
    <p>
      <a class="button" href="<?php echo esc_url(admin_url('admin-post.php?action=hitokoto_export&format=json')); ?>">导出 JSON</a>
      <a class="button" href="<?php echo esc_url(admin_url('admin-post.php?action=hitokoto_export&format=csv')); ?>">导出 CSV</a>
    </p>
    <p>也可用 REST：<code>/wp-json/hitokoto/v1/export?format=json</code>（需管理员登录）。</p>
  </div>
  <?php
}
add_action('admin_post_hitokoto_import','hitokoto_handle_import');
add_action('admin_post_hitokoto_export','hitokoto_handle_export');

/** 导入处理 */
function hitokoto_handle_import() {
  if (!current_user_can('manage_options')) wp_die('权限不足');
  if (!isset($_POST['hitokoto_import_nonce']) || !wp_verify_nonce($_POST['hitokoto_import_nonce'],'hitokoto_import')) wp_die('非法请求');
  if (!isset($_FILES['hitokoto_file'])) wp_die('未选择文件');

  $replace = !empty($_POST['replace']);
  $file = $_FILES['hitokoto_file'];
  if ($file['error'] !== UPLOAD_ERR_OK) wp_die('上传失败：' . $file['error']);

  $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
  $tmp = $file['tmp_name'];
  $content = file_get_contents($tmp);

  $items = [];
  if ($ext === 'json' || (strlen($content) && ($content[0] === '[' || $content[0] === '{'))) {
    $decoded = json_decode($content, true);
    if (json_last_error() === JSON_ERROR_NONE) {
      if (isset($decoded[0])) $items = $decoded; else $items = [$decoded];
    } else {
      $lines = preg_split('/\r\n|\r|\n/', $content);
      foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '') continue;
        $obj = json_decode($line, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($obj)) $items[] = $obj;
      }
    }
  } else { // CSV 或 TXT
    $lines = preg_split('/\r\n|\r|\n/', $content);
    $header = null;
    foreach ($lines as $line) {
      $line = trim($line);
      if ($line === '') continue;
      $row = str_getcsv($line);
      if (!$header) { $header = array_map('trim', $row); continue; }
      $item = [];
      foreach ($header as $i => $col) { $item[$col] = $row[$i] ?? ''; }
      $items[] = $item;
    }
  }

  $inserted = 0; $updated = 0; $failed = 0;
  foreach ($items as $it) {
    $result = hitokoto_insert_or_update($it, $replace);
    if ($result === false) $failed++;
    elseif ($result === 'updated') $updated++;
    else $inserted++;
  }

  $msg = sprintf('导入完成：新增 %d，更新 %d，失败 %d。', $inserted, $updated, $failed);
  $redirect = add_query_arg(['import_result' => urlencode($msg)], admin_url('edit.php?post_type=hitokoto&page=hitokoto-import-export'));
  wp_redirect($redirect); exit;
}

/** 插入或更新（按内容哈希去重） */
function hitokoto_insert_or_update($item, $replace) {
  $content = trim((string)($item['hitokoto'] ?? ''));
  if ($content === '') return false;
  $hash = md5(trim(wp_strip_all_tags($content)));

  // 查重（按哈希）
  $existing = get_posts([
    'post_type' => 'hitokoto',
    'post_status' => 'any',
    'meta_query' => [[
      'key' => 'hitokoto_hash',
      'value' => $hash,
      'compare' => '='
    ]],
    'posts_per_page' => 1,
    'fields' => 'ids',
    'no_found_rows' => true
  ]);
  $existing_id = $existing ? $existing[0] : 0;

  $postarr = [
    'post_type' => 'hitokoto',
    'post_status' => 'publish',
    'post_title' => wp_trim_words($content, 8, ''),
    'post_content' => $content
  ];

  if ($existing_id && $replace) {
    $postarr['ID'] = $existing_id;
    $post_id = wp_update_post($postarr, true);
    if (is_wp_error($post_id)) return false;
    $status = 'updated';
  } elseif ($existing_id && !$replace) {
    return 'updated'; // 视作已存在，不重复插入
  } else {
    $post_id = wp_insert_post($postarr, true);
    if (is_wp_error($post_id)) return false;
    $status = 'inserted';
  }

  update_post_meta($post_id,'hitokoto_hash',$hash);
  update_post_meta($post_id,'hitokoto_from',     sanitize_text_field($item['from']      ?? ''));
  update_post_meta($post_id,'hitokoto_from_who', sanitize_text_field($item['from_who']  ?? ''));
  update_post_meta($post_id,'hitokoto_lang',     sanitize_text_field($item['lang']      ?? ''));

  $type = sanitize_title($item['type'] ?? '');
  if ($type !== '') {
    $term = term_exists($type, 'hitokoto_type');
    if (!$term || is_wp_error($term)) {
      $term = wp_insert_term($type, 'hitokoto_type', ['slug' => $type]);
    }
    if (!is_wp_error($term)) {
      wp_set_object_terms($post_id, $type, 'hitokoto_type', false);
    }
  }
  return $status === 'updated' ? 'updated' : $post_id;
}

/** 导出处理（后台） */
function hitokoto_handle_export() {
  if (!current_user_can('manage_options')) wp_die('权限不足');
  $format = isset($_GET['format']) ? sanitize_text_field($_GET['format']) : 'json';
  $type   = isset($_GET['type'])   ? sanitize_text_field($_GET['type'])   : '';
  $lang   = isset($_GET['lang'])   ? sanitize_text_field($_GET['lang'])   : '';

  $ids = hitokoto_get_id_pool($type, $lang);

  if ($format === 'csv') {
    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename=hitokoto.csv');
    echo "hitokoto,from,from_who,type,lang\r\n";
    foreach ($ids as $id) {
      $p = get_post($id);
      $content = str_replace(["\r","\n"], ' ', trim(wp_strip_all_tags($p->post_content)));
      $from     = get_post_meta($id, 'hitokoto_from', true);
      $from_who = get_post_meta($id, 'hitokoto_from_who', true);
      $lang_val = get_post_meta($id, 'hitokoto_lang', true);
      $terms = wp_get_post_terms($id, 'hitokoto_type');
      $type_slug_actual = isset($terms[0]) ? $terms[0]->slug : '';
      $row = [
        $content,
        $from ?: '',
        $from_who ?: '',
        $type_slug_actual,
        $lang_val ?: ''
      ];
      $row = array_map(function($v){ return '"' . str_replace('"','""',$v) . '"'; }, $row);
      echo implode(',', $row) . "\r\n";
    }
    exit;
  } else {
    header('Content-Type: application/json; charset=UTF-8');
    $data = [];
    foreach ($ids as $id) {
      $p = get_post($id);
      $content = trim(wp_strip_all_tags($p->post_content));
      $from     = get_post_meta($id, 'hitokoto_from', true);
      $from_who = get_post_meta($id, 'hitokoto_from_who', true);
      $lang_val = get_post_meta($id, 'hitokoto_lang', true);
      $terms = wp_get_post_terms($id, 'hitokoto_type');
      $type_slug_actual = isset($terms[0]) ? $terms[0]->slug : '';
      $data[] = [
        'id'       => (int)$id,
        'hitokoto' => $content,
        'from'     => $from ?: '',
        'from_who' => $from_who ?: '',
        'type'     => $type_slug_actual,
        'lang'     => $lang_val ?: '',
        'length'   => hitokoto_strlen($content),
      ];
    }
    echo wp_json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
  }
}